<?php
session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';

// Check if user has any supplier-related permissions
$supplierPermissions = [
    'view_suppliers', 'manage_suppliers', 'add_supplier', 'edit_supplier', 'delete_supplier',
    'search_suppliers', 'process_supplier_payment', 'add_supplier_item'
];

if (!hasAnyPermission($supplierPermissions) && !$isSuperAdmin) {
    $_SESSION['error'] = "You don't have permission to access supplier management.";
    header('Location: ../pages/dashboard.php');
    exit;
}

$activePage = 'suppliers';
$success = '';
$error = '';

// Handle supplier deletion
if (isset($_POST['delete_supplier']) && (hasPermission('manage_suppliers') || $isSuperAdmin)) {
    try {
        $stmt = $pdo->prepare("DELETE FROM suppliers WHERE id = ?");
        $stmt->execute([$_POST['supplier_id']]);
        $success = "Supplier deleted successfully!";
    } catch (PDOException $e) {
        $error = "Error deleting supplier: " . $e->getMessage();
    }
}

// Handle supplier addition/update
if (isset($_POST['save_supplier']) && (hasPermission('manage_suppliers') || $isSuperAdmin)) {
    try {
        $name = trim($_POST['name']);
        $contact_person = trim($_POST['contact_person']);
        $phone = trim($_POST['phone']);
        $email = trim($_POST['email']);
        $address = trim($_POST['address']);
        $tax_number = trim($_POST['tax_number']);

        if (empty($name)) {
            throw new Exception("Supplier name is required");
        }

        if (isset($_POST['supplier_id']) && !empty($_POST['supplier_id'])) {
            // Update existing supplier
            $stmt = $pdo->prepare("
                UPDATE suppliers 
                SET name = ?, 
                    contact_person = ?, 
                    phone = ?, 
                    email = ?, 
                    address = ?, 
                    tax_number = ? 
                WHERE id = ?
            ");
            $stmt->execute([$name, $contact_person, $phone, $email, $address, $tax_number, $_POST['supplier_id']]);
            $success = "Supplier updated successfully!";
        } else {
            // Add new supplier
            $stmt = $pdo->prepare("
                INSERT INTO suppliers (
                    name, 
                    contact_person, 
                    phone, 
                    email, 
                    address, 
                    tax_number
                ) VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$name, $contact_person, $phone, $email, $address, $tax_number]);
            $success = "Supplier added successfully!";
        }

        // Log the action
        logSystemAction(
            isset($_POST['supplier_id']) ? 'supplier_updated' : 'supplier_created',
            isset($_POST['supplier_id']) ? "Updated supplier: {$name}" : "Added new supplier: {$name}"
        );

        // Redirect to refresh the page
        header("Location: suppliers.php?success=" . urlencode($success));
        exit;
    } catch (Exception $e) {
        $error = "Error saving supplier: " . $e->getMessage();
    }
}

// Get all suppliers with their items and payment details (including completed orders)
$stmt = $pdo->query("
    SELECT 
        s.*,
        (COALESCE(sr_counts.total_items, 0) + COALESCE(so_counts.total_order_items, 0)) as total_items,
        (COALESCE(sr_counts.paid_items, 0) + COALESCE(so_counts.completed_order_items, 0)) as paid_items,
        (COALESCE(sr_totals.total_amount, 0) + COALESCE(so_totals.total_order_amount, 0)) as total_amount,
        (COALESCE(sp_totals.total_paid, 0) + COALESCE(so_totals.completed_order_amount, 0)) as paid_amount,
        GREATEST(
            COALESCE(sp_dates.last_payment_date, '1970-01-01'), 
            COALESCE(so_dates.last_order_date, '1970-01-01')
        ) as last_payment_date,
        LEAST(
            COALESCE(sr_dates.oldest_pending_date, '9999-12-31'), 
            COALESCE(so_dates.oldest_pending_date, '9999-12-31')
        ) as oldest_pending_date
    FROM suppliers s
             LEFT JOIN (
             SELECT 
                 supplier_id,
                 COUNT(*) as total_items,
                 SUM(CASE WHEN payment_status = 'paid' THEN 1 ELSE 0 END) as paid_items
             FROM supplier_receipts 
             GROUP BY supplier_id
         ) sr_counts ON s.id = sr_counts.supplier_id
             LEFT JOIN (
             SELECT 
                 supplier_id,
                 SUM(quantity * unit_price) as total_amount
             FROM supplier_receipts 
             GROUP BY supplier_id
         ) sr_totals ON s.id = sr_totals.supplier_id
             LEFT JOIN (
             SELECT 
                 supplier_id,
                 SUM(amount) as total_paid
             FROM supplier_payments 
             GROUP BY supplier_id
         ) sp_totals ON s.id = sp_totals.supplier_id
             LEFT JOIN (
             SELECT 
                 supplier_id,
                 MAX(created_at) as last_payment_date
             FROM supplier_payments 
             GROUP BY supplier_id
         ) sp_dates ON s.id = sp_dates.supplier_id
             LEFT JOIN (
             SELECT 
                 supplier_id,
                 MIN(CASE WHEN payment_status = 'pending' THEN created_at END) as oldest_pending_date
             FROM supplier_receipts 
             GROUP BY supplier_id
         ) sr_dates ON s.id = sr_dates.supplier_id
    LEFT JOIN (
        SELECT 
            so.supplier_id,
            COUNT(soi.id) as total_order_items,
            SUM(CASE WHEN so.status = 'completed' THEN 1 ELSE 0 END) as completed_order_items
        FROM supplier_orders so
        JOIN supplier_order_items soi ON so.id = soi.order_id
        GROUP BY so.supplier_id
    ) so_counts ON s.id = so_counts.supplier_id
    LEFT JOIN (
        SELECT 
            so.supplier_id,
            SUM(soi.quantity * COALESCE(soi.unit_price, 0)) as total_order_amount,
            SUM(CASE WHEN so.status = 'completed' THEN soi.quantity * COALESCE(soi.unit_price, 0) ELSE 0 END) as completed_order_amount
        FROM supplier_orders so
        JOIN supplier_order_items soi ON so.id = soi.order_id
        GROUP BY so.supplier_id
    ) so_totals ON s.id = so_totals.supplier_id
    LEFT JOIN (
        SELECT 
            supplier_id,
            MAX(CASE WHEN status = 'completed' THEN created_at END) as last_order_date,
            MIN(CASE WHEN status = 'pending' THEN created_at END) as oldest_pending_date
        FROM supplier_orders 
        GROUP BY supplier_id
    ) so_dates ON s.id = so_dates.supplier_id
    GROUP BY s.id
    ORDER BY s.name
");
$suppliers = $stmt->fetchAll();

// Get all products for the Create New Receipt modal
$stmt = $pdo->query("SELECT id, name FROM products WHERE status = 'active' ORDER BY name");
$products = $stmt->fetchAll();

// Get all items for each supplier (receipts and manual adjustments)
$supplier_items = [];
foreach ($suppliers as $supplier) {
    // Supplier Receipts
    $stmt = $pdo->prepare("
        SELECT 
            sr.id,
            sr.created_at,
            p.name as product_name,
            sr.batch_number,
            sr.quantity,
            sr.unit_price,
            (sr.quantity * sr.unit_price) as total,
            sr.payment_status,
            u.username as entered_by,
            'Receipt' as source,
            sr.serial_numbers,
            sr.payment_status as payment_status_raw,
            sr.payment_status as payment_status_display
        FROM supplier_receipts sr
        JOIN products p ON sr.product_id = p.id
        JOIN users u ON sr.created_by = u.id
        WHERE sr.supplier_id = ?
        ORDER BY sr.created_at DESC
    ");
    $stmt->execute([$supplier['id']]);
    $receipts = $stmt->fetchAll();

    // Manual Stock Adjustments
    $stmt = $pdo->prepare("
        SELECT 
            sa.id,
            sa.adjusted_at as created_at,
            p.name as product_name,
            NULL as batch_number,
            sa.quantity,
            NULL as unit_price,
            NULL as total,
            NULL as payment_status,
            u.username as entered_by,
            'Manual Adjustment' as source,
            NULL as serial_numbers,
            NULL as payment_status_raw,
            NULL as payment_status_display
        FROM stock_adjustments sa
        JOIN products p ON sa.product_id = p.id
        JOIN users u ON sa.adjusted_by = u.id
        WHERE sa.supplier_id = ?
        ORDER BY sa.adjusted_at DESC
    ");
    $stmt->execute([$supplier['id']]);
    $adjustments = $stmt->fetchAll();

    // Completed Supplier Orders
    $stmt = $pdo->prepare("
        SELECT 
            CONCAT('SO-', so.id, '-', soi.id) as id,
            so.created_at,
            p.name as product_name,
            CONCAT('Order #', so.id) as batch_number,
            soi.quantity,
            soi.unit_price,
            (soi.quantity * COALESCE(soi.unit_price, 0)) as total,
            'completed' as payment_status,
            u.username as entered_by,
            'Supplier Order' as source,
            NULL as serial_numbers,
            'completed' as payment_status_raw,
            'Completed' as payment_status_display
        FROM supplier_orders so
        JOIN supplier_order_items soi ON so.id = soi.order_id
        JOIN products p ON soi.product_id = p.id
        JOIN users u ON so.created_by = u.id
        WHERE so.supplier_id = ? AND so.status = 'completed'
        ORDER BY so.created_at DESC
    ");
    $stmt->execute([$supplier['id']]);
    $orders = $stmt->fetchAll();

    // Merge and sort by created_at DESC
    $all_items = array_merge($receipts, $adjustments, $orders);
    usort($all_items, function($a, $b) {
        return strtotime($b['created_at']) <=> strtotime($a['created_at']);
    });
    $supplier_items[$supplier['id']] = $all_items;
}

// Get store information
$stmt = $pdo->prepare("
    SELECT setting_key, setting_value 
    FROM store_settings 
    WHERE setting_group = 'store_info' 
    AND setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'currency', 'store_logo', 'website')
");
$stmt->execute();
$storeSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Get notification settings
$stmt = $pdo->prepare("
    SELECT setting_key, setting_value 
    FROM store_settings 
    WHERE setting_key IN ('enable_sms', 'enable_email', 'enable_whatsapp')
");
$stmt->execute();
$notificationSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Check for success message in URL
if (isset($_GET['success'])) {
    $success = $_GET['success'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Suppliers - EDDUH COMPUTERS</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css" />
    <style>
        body { background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%); font-family: 'Inter', sans-serif; }
        .card { border-radius: 16px; box-shadow: 0 2px 12px rgba(37,99,235,0.06); border: none; }
        .card-header { background: none; border-bottom: none; }
        .btn-primary { border-radius: 8px; font-weight: 600; }
        .table { background: #fff; border-radius: 12px; overflow: hidden; }
        .table th, .table td { vertical-align: middle; padding: 6px; }
        .table-striped > tbody > tr:nth-of-type(odd) { background-color: #f3f4f6; }
        .table thead th { 
            background-color: #f8fafc;
            border-bottom: 2px solid #e2e8f0;
            color: #1e40af;
            font-weight: 600;
        }
        .table tbody td {
            color: #334155;
        }
        .btn-sm {
            padding: 1px 4px;
            font-size: 0.6rem;
            margin: 0 0.5px;
        }
        .modal-content { border-radius: 16px; }
        .form-label { font-weight: 600; color: #2563eb; }
        .modal-title { color: #2563eb; font-weight: 700; }
        .supplier-header:hover {
            background-color: #f8f9fa;
        }
        .supplier-header.expanded {
            background-color: #e3f2fd;
        }
        .supplier-toggle {
            transition: transform 0.2s;
        }
        .supplier-details td {
            padding: 0 !important;
        }
        .supplier-details .table {
            margin-bottom: 0;
        }
        .modal-content {
            border: none;
            border-radius: 15px;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
            max-height: 90vh; /* Limit modal height to 90% of viewport height */
        }
        .modal-header {
            border-radius: 15px 15px 0 0;
            padding: 1rem;
        }
        .modal-body {
            padding: 1.5rem;
            max-height: calc(90vh - 130px); /* Subtract header and footer height */
            overflow-y: auto;
        }
        .modal-footer {
            border-radius: 0 0 15px 15px;
            padding: 1rem;
            background: #f8f9fa;
        }
        .card {
            transition: all 0.3s ease;
        }
        .card:hover {
            transform: translateY(-2px);
        }
        .form-control, .form-select {
            border-radius: 8px;
            border: 1px solid #e0e0e0;
            padding: 0.75rem 1rem;
        }
        .form-control:focus, .form-select:focus {
            border-color: #0d6efd;
            box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.15);
        }
        .table {
            margin-bottom: 0;
        }
        .table th {
            font-weight: 600;
            color: #495057;
        }
        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-primary {
            background: linear-gradient(45deg, #0d6efd, #0a58ca);
            border: none;
        }
        .btn-primary:hover {
            background: linear-gradient(45deg, #0a58ca, #084298);
            transform: translateY(-1px);
        }
        .btn-light {
            background: #f8f9fa;
            border: 1px solid #e0e0e0;
        }
        .btn-light:hover {
            background: #e9ecef;
        }
        #paymentDistributionTable tbody tr {
            transition: all 0.2s ease;
        }
        #paymentDistributionTable tbody tr:hover {
            background-color: #f8f9fa;
        }
        .form-label {
            font-weight: 500;
            color: #495057;
        }
        .text-muted {
            color: #6c757d !important;
        }
        .shadow-sm {
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075) !important;
        }
        /* Make payment distribution table scrollable */
        .payment-distribution-container {
            max-height: 300px;
            overflow-y: auto;
            border: 1px solid #e9ecef;
            border-radius: 8px;
        }
        .payment-distribution-container table {
            margin-bottom: 0;
        }
        .payment-distribution-container thead th {
            position: sticky;
            top: 0;
            background: #f8f9fa;
            z-index: 1;
        }
        /* Add a subtle shadow to indicate scrollable content */
        .payment-distribution-container::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: 20px;
            background: linear-gradient(to bottom, rgba(255,255,255,0), rgba(255,255,255,1));
            pointer-events: none;
        }
        
        /* Disabled notification card styling */
        .notification-card-disabled {
            opacity: 0.5 !important;
            filter: grayscale(1) !important;
            pointer-events: none;
        }
        
        .notification-card-disabled .form-check-input {
            opacity: 0.5;
        }
        
        .notification-card-disabled .form-check-label {
            color: #6c757d !important;
        }
        
        .notification-card-disabled .card-body {
            background-color: #f8f9fa;
        }
        
        /* Modern gradient backgrounds */
        .bg-gradient-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        
        .bg-gradient-light {
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
        }
        
        /* Enhanced form controls */
        .form-control:focus, .form-select:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        
        /* Modern card hover effects */
        .card {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .card:hover {
            transform: translateY(-4px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.1) !important;
        }
        
        /* Enhanced buttons */
        .btn {
            border-radius: 12px;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
        }
        
        .btn-primary:hover {
            background: linear-gradient(135deg, #5a6fd8 0%, #6a4190 100%);
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(102, 126, 234, 0.3);
        }
        
        /* Modern table styling */
        .table th {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border: none;
            font-weight: 600;
            color: #495057;
        }
        
        .table td {
            border: none;
            padding: 1rem 0.75rem;
        }
        
        .table tbody tr:hover {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        }
        
        /* Enhanced modal styling */
        .modal-content {
            border-radius: 20px;
            overflow: hidden;
        }

        .modal-header {
            padding: 1.5rem 2rem;
        }
        
        .modal-body {
            padding: 2rem;
        }
        
        .modal-footer {
            padding: 1.5rem 2rem;
        }
        
        /* Icon containers */
        .bg-success.bg-opacity-10 {
            background-color: rgba(25, 135, 84, 0.1) !important;
        }
        
        .bg-info.bg-opacity-10 {
            background-color: rgba(13, 202, 240, 0.1) !important;
        }
        
        .bg-primary.bg-opacity-10 {
            background-color: rgba(13, 110, 253, 0.1) !important;
        }
        
        .bg-warning.bg-opacity-10 {
            background-color: rgba(255, 193, 7, 0.1) !important;
        }
        
        /* Enhanced input groups */
        .input-group-text {
            border: none;
            font-weight: 600;
        }
        
        /* Payment distribution container */
        .payment-distribution-container {
            border-radius: 12px;
            overflow: hidden;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <!-- Main Content -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">Suppliers</h4>
                        <div class="d-flex align-items-center">
                            <div class="input-group me-3" style="width: 300px;">
                                <span class="input-group-text"><i class="fas fa-search"></i></span>
                                <input type="text" id="supplierSearch" class="form-control" placeholder="Search suppliers...">
                            </div>
                            <?php if (hasPermission('manage_suppliers')): ?>
                            <div>
                                <a href="viewsuppliers.php" class="btn btn-info me-2">
                                    <i class="fas fa-list me-2"></i>View Suppliers
                                </a>
                                <a href="supplier_order.php" class="btn btn-warning me-2">
                                    <i class="fas fa-shopping-cart me-2"></i>Order Items
                                </a>
                                <button type="button" class="btn btn-primary me-2" data-bs-toggle="modal" data-bs-target="#supplierModal">
                                    <i class="fas fa-plus me-2"></i>Add Supplier
                                </button>
                                <a href="receivesupplieritem.php" class="btn btn-info">
                                    <i class="fas fa-receipt"></i> Supplier Receipts
                                </a>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success"><?php echo $success; ?></div>
                        <?php endif; ?>
                        <?php if ($error): ?>
                            <div class="alert alert-danger"><?php echo $error; ?></div>
                        <?php endif; ?>

                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Supplier</th>
                                        <th>Contact</th>
                                        <th>Total Items</th>
                                        <th>Total Amount</th>
                                        <th>Total Paid</th>
                                        <th>Total Balance</th>
                                        <th>Payment Dates</th>
                                        <th>Payment Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <tr class="supplier-header" data-supplier-id="<?php echo $supplier['id']; ?>" style="cursor: pointer;">
                                            <td>
                                                <i class="fas fa-chevron-right me-2 supplier-toggle"></i>
                                                <?php echo htmlspecialchars($supplier['name']); ?>
                                            </td>
                                            <td>
                                                <?php echo htmlspecialchars($supplier['contact_person']); ?><br>
                                                <small class="text-muted"><?php echo htmlspecialchars($supplier['phone']); ?></small>
                                            </td>
                                            <td><?php echo $supplier['total_items']; ?></td>
                                            <td><?php echo number_format($supplier['total_amount'], 2); ?></td>
                                            <td><?php echo number_format($supplier['paid_amount'], 2); ?></td>
                                            <td><?php echo number_format($supplier['total_amount'] - $supplier['paid_amount'], 2); ?></td>
                                            <td>
                                                <?php if ($supplier['last_payment_date']): ?>
                                                    <small class="text-success">Last Paid: <?php echo date('Y-m-d', strtotime($supplier['last_payment_date'])); ?></small><br>
                                                <?php endif; ?>
                                                <?php if ($supplier['oldest_pending_date']): ?>
                                                    <small class="text-warning">Oldest Pending: <?php echo date('Y-m-d', strtotime($supplier['oldest_pending_date'])); ?></small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php
                                                $paid_count = $supplier['paid_items'];
                                                $total_count = $supplier['total_items'];
                                                $paid_amount = $supplier['paid_amount'];
                                                $total_amount = $supplier['total_amount'];
                                                echo "{$paid_count}/{$total_count} Items<br>";
                                                echo "<small class='text-muted'>" . number_format($paid_amount, 2) . " / " . number_format($total_amount, 2) . " Paid</small>";
                                                ?>
                                            </td>
                                            <td>
                                                <div class="btn-group">
                                                    <button type="button" class="btn btn-success btn-sm pay-btn" 
                                                        data-supplier-id="<?php echo $supplier['id']; ?>"
                                                        data-supplier-name="<?php echo htmlspecialchars($supplier['name']); ?>"
                                                        data-total-amount="<?php echo $supplier['total_amount']; ?>"
                                                        data-paid-amount="<?php echo $supplier['paid_amount']; ?>"
                                                        data-items='<?php 
                                                            $items_data = array_map(function($item) {
                                                            return [
                                                                'id' => $item['id'],
                                                                'created_at' => $item['created_at'],
                                                                'product_name' => $item['product_name'],
                                                                'quantity' => $item['quantity'],
                                                                'unit_price' => $item['unit_price'],
                                                                'payment_status' => $item['payment_status']
                                                            ];
                                                            }, $supplier_items[$supplier['id']]);
                                                            echo htmlspecialchars(json_encode($items_data, JSON_HEX_APOS | JSON_HEX_QUOT));
                                                        ?>'>
                                                        <i class="fas fa-money-bill"></i> Pay
                                                    </button>
                                                    <button type="button" class="btn btn-info btn-sm" onclick="event.stopPropagation(); editSupplier(<?php echo htmlspecialchars(json_encode($supplier, JSON_HEX_APOS | JSON_HEX_QUOT)); ?>)">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-danger btn-sm" onclick="event.stopPropagation(); confirmDeleteSupplier(<?php echo $supplier['id']; ?>, '<?php echo addslashes($supplier['name']); ?>', <?php echo number_format($supplier['total_amount'], 2, '.', ''); ?>, <?php echo number_format($supplier['paid_amount'], 2, '.', ''); ?>)">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-secondary btn-sm" onclick="event.stopPropagation(); openPrintModal(<?php echo $supplier['id']; ?>, '<?php echo addslashes($supplier['name']); ?>');">
                                                        <i class="fas fa-print"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                        <tr class="supplier-details" data-supplier-id="<?php echo $supplier['id']; ?>" style="display: none;">
                                            <td colspan="7">
                                                <div class="table-responsive">
                                                    <table class="table table-sm table-bordered">
                                                        <thead>
                                                            <tr>
                                                                <th>Date</th>
                                                                <th>Product</th>
                                                                <th>Batch Number</th>
                                                                <th>Qty</th>
                                                                <th>Price</th>
                                                                <th>Total</th>
                                                                <th>Source</th>
                                                                <th>Payment Status</th>
                                                                <th>Entered By</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php foreach ($supplier_items[$supplier['id']] as $item): ?>
                                                                <tr>
                                                                    <td><?php echo date('Y-m-d H:i', strtotime($item['created_at'])); ?></td>
                                                                    <td><?php echo htmlspecialchars($item['product_name']); ?></td>
                                                                    <td><?php echo htmlspecialchars($item['batch_number'] ?? ''); ?></td>
                                                                    <td><?php echo number_format($item['quantity'], 2); ?></td>
                                                                    <td><?php echo $item['unit_price'] !== null ? number_format($item['unit_price'], 2) : '-'; ?></td>
                                                                    <td><?php echo $item['total'] !== null ? number_format($item['total'], 2) : '-'; ?></td>
                                                                    <td><span class="badge bg-<?php 
                                                                        echo $item['source'] === 'Receipt' ? 'primary' : 
                                                                            ($item['source'] === 'Supplier Order' ? 'success' : 'warning'); 
                                                                    ?>"><?php echo $item['source']; ?></span></td>
                                                                    <td>
                                                                        <?php if ($item['source'] === 'Receipt'): ?>
                                                                            <span class="badge bg-<?php 
                                                                                echo $item['payment_status_raw'] === 'paid' ? 'success' : 
                                                                                    ($item['payment_status_raw'] === 'credit' ? 'warning' : 'secondary'); 
                                                                            ?>">
                                                                                <?php echo $item['payment_status_display']; ?>
                                                                            </span>
                                                                        <?php elseif ($item['source'] === 'Supplier Order'): ?>
                                                                            <span class="badge bg-success">Completed</span>
                                                                        <?php else: ?>
                                                                            <span class="badge bg-info">N/A</span>
                                                                        <?php endif; ?>
                                                                    </td>
                                                                    <td><?php echo htmlspecialchars($item['entered_by']); ?></td>
                                                                </tr>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Supplier Modal -->
    <div class="modal fade" id="supplierModal" tabindex="-1" aria-labelledby="supplierModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="supplierModalLabel">Add Supplier</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="post" class="needs-validation" novalidate>
                    <div class="modal-body">
                        <input type="hidden" name="supplier_id" id="supplier_id">
                        <div class="mb-3">
                            <label for="name" class="form-label">Supplier Name *</label>
                            <input type="text" class="form-control" id="name" name="name" required>
                            <div class="invalid-feedback">Please enter supplier name.</div>
                        </div>
                        <div class="mb-3">
                            <label for="contact_person" class="form-label">Contact Person</label>
                            <input type="text" class="form-control" id="contact_person" name="contact_person">
                        </div>
                        <div class="mb-3">
                            <label for="phone" class="form-label">Phone</label>
                            <input type="tel" class="form-control" id="phone" name="phone">
                        </div>
                        <div class="mb-3">
                            <label for="email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="email" name="email">
                        </div>
                        <div class="mb-3">
                            <label for="address" class="form-label">Address</label>
                            <textarea class="form-control" id="address" name="address" rows="2"></textarea>
                        </div>
                        <div class="mb-3">
                            <label for="tax_number" class="form-label">Tax Number</label>
                            <input type="text" class="form-control" id="tax_number" name="tax_number">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="save_supplier" class="btn btn-primary">Save Supplier</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Payment Modal -->
    <div class="modal fade" id="paymentModal" tabindex="-1" aria-labelledby="paymentModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content border-0 shadow-lg">
                <div class="modal-header bg-gradient-primary text-white border-0">
                    <h5 class="modal-title" id="paymentModalLabel">
                        <i class="fas fa-credit-card me-2"></i>Payment Details
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-3" style="background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);">
                    <form id="paymentForm" method="post">
                        <input type="hidden" name="supplier_id" id="payment_supplier_id">
                        
                                                <!-- Balance Section -->
                        <div class="row mb-3">
                            <div class="col-12">
                                <div class="card border border-primary" style="background: #f8f9ff;">
                                    <div class="card-body text-center p-3">
                                        <div class="mb-2">
                                            <i class="fas fa-wallet fa-lg text-primary"></i>
                                        </div>
                                        <label class="form-label small fw-bold text-uppercase text-muted mb-1">Total Balance</label>
                                        <h4 class="mb-1 fw-bold text-primary" id="total_balance">0.00</h4>
                                        <small class="text-muted d-block">Amount owed</small>
                                        <div class="mt-1 small text-muted" id="balance_details">
                                            <span id="balance_calculation">Calculating...</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Payment Method Section -->
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <div class="card h-100 border border-secondary" style="background: white;">
                                    <div class="card-body p-2">
                                        <div class="mb-2">
                                            <label class="form-label fw-bold text-dark small">
                                                <i class="fas fa-credit-card me-1 text-primary"></i>Payment Method
                                            </label>
                                        </div>
                                        <select class="form-select border border-secondary form-select-sm" name="payment_method" id="payment_method" required onchange="handlePaymentMethodChange()">
                                            <option value="">Choose payment method...</option>
                                            <option value="cash">💵 Cash</option>
                                            <option value="mpesa">📱 M-PESA</option>
                                            <option value="cheque">🏦 Cheque</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card h-100 border border-info" id="mpesaSection" style="display: none; background: #f0f9ff;">
                                    <div class="card-body p-2">
                                        <div class="d-flex align-items-center mb-2">
                                            <div class="bg-info bg-opacity-10 p-1 rounded me-2">
                                                <i class="fas fa-mobile-alt text-info"></i>
                                            </div>
                                            <h6 class="mb-0 fw-bold text-info small">M-PESA Details</h6>
                                        </div>
                                        <div class="mb-1">
                                            <label class="form-label fw-semibold small text-dark">Transaction Code</label>
                                            <input type="text" class="form-control border border-info form-control-sm" name="mpesa_code" id="mpesa_code" placeholder="Enter M-PESA transaction code">
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card h-100 border border-warning" id="chequeSection" style="display: none; background: #fffbf0;">
                                    <div class="card-body p-2">
                                        <div class="d-flex align-items-center mb-2">
                                            <div class="bg-warning bg-opacity-10 p-1 rounded me-2">
                                                <i class="fas fa-money-check text-warning"></i>
                                            </div>
                                            <h6 class="mb-0 fw-bold text-warning small">Cheque Details</h6>
                                        </div>
                                        <div class="mb-1">
                                            <label class="form-label fw-semibold small text-dark">Cheque Number</label>
                                            <input type="text" class="form-control border border-warning form-control-sm" name="cheque_number" id="cheque_number" placeholder="Enter cheque number">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Amount and Notes Section -->
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="card border border-primary h-100" style="background: white;">
                                    <div class="card-body p-2">
                                        <div class="mb-2">
                                            <label class="form-label fw-bold text-dark small">
                                                <i class="fas fa-money-bill-wave me-1 text-primary"></i>Payment Amount
                                            </label>
                                        </div>
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text bg-primary text-white border-primary">
                                                <i class="fas fa-coins"></i>
                                            </span>
                                            <input type="number" class="form-control border-primary" name="amount" id="payment_amount" step="any" required oninput="updatePaymentPreview()" onchange="updatePaymentDistribution()" placeholder="0.00">
                                        </div>
                                        <div class="invalid-feedback">Please enter a valid amount.</div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="card border border-secondary h-100" style="background: white;">
                                    <div class="card-body p-2">
                                        <div class="mb-2">
                                            <label class="form-label fw-bold text-dark small">
                                                <i class="fas fa-sticky-note me-1 text-secondary"></i>Payment Notes
                                            </label>
                                        </div>
                                        <textarea class="form-control border-secondary form-control-sm" name="payment_notes" id="payment_notes" rows="2" placeholder="Optional notes about this payment..." style="resize: none;"></textarea>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Message Preview and Notification Options -->
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="card border border-info h-100" style="background: #f0f9ff;">
                                    <div class="card-body p-1">
                                        <div class="d-flex align-items-center mb-1">
                                            <div class="bg-info bg-opacity-10 p-1 rounded me-1">
                                                <i class="fas fa-paper-plane text-info"></i>
                                            </div>
                                            <h6 class="mb-0 fw-bold text-info small">Message Preview</h6>
                                        </div>
                                        <div id="sendingMessagePreview" class="bg-white p-1 rounded border border-info border-opacity-20" style="min-height: 30px; font-size: 0.8em; line-height: 1.3; color: #333;">
                                            <i class="fas fa-info-circle me-1 text-muted"></i>
                                            <span class="text-muted">Fill in payment details to preview the message.</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="card border border-secondary h-100" style="background: white;">
                                    <div class="card-body p-1">
                                        <div class="d-flex align-items-center mb-1">
                                            <div class="bg-secondary bg-opacity-10 p-1 rounded me-1">
                                                <i class="fas fa-bell text-secondary"></i>
                                            </div>
                                            <h6 class="mb-0 fw-bold text-secondary small">Notification Options</h6>
                                        </div>
                                        <div class="row">
                                            <div class="col-12 mb-1">
                                                <div class="form-check p-1 border rounded">
                                                    <input class="form-check-input" type="checkbox" name="send_sms" id="send_sms" value="1">
                                                    <label class="form-check-label fw-semibold small" for="send_sms">
                                                        <i class="fas fa-sms me-1 text-primary"></i>
                                                        SMS Notification
                                                    </label>
                                                </div>
                                            </div>
                                            <div class="col-12 mb-1">
                                                <div class="form-check p-1 border rounded">
                                                    <input class="form-check-input" type="checkbox" name="send_whatsapp" id="send_whatsapp" value="1">
                                                    <label class="form-check-label fw-semibold small" for="send_whatsapp">
                                                        <i class="fab fa-whatsapp me-1 text-success"></i>
                                                        WhatsApp Message
                                                    </label>
                                                </div>
                                            </div>
                                            <div class="col-12 mb-1">
                                                <div class="form-check p-1 border rounded">
                                                    <input class="form-check-input" type="checkbox" name="send_email" id="send_email" value="1">
                                                    <label class="form-check-label fw-semibold small" for="send_email">
                                                        <i class="fas fa-envelope me-1 text-info"></i>
                                                        Email Notification
                                                    </label>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="mt-1">
                                            <small class="text-muted">
                                                <i class="fas fa-info-circle me-1"></i>
                                                Notifications will be sent to the supplier's contact person with payment details.
                                            </small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Payment Distribution Section -->
                        <div class="card border border-primary" style="background: white;">
                            <div class="card-body p-2">
                                <div class="d-flex align-items-center mb-2">
                                    <div class="bg-primary bg-opacity-10 p-1 rounded me-2">
                                        <i class="fas fa-chart-pie text-primary"></i>
                                    </div>
                                    <h6 class="mb-0 fw-bold text-primary small">Payment Distribution</h6>
                                </div>
                                <div class="payment-distribution-container border border-primary rounded" style="background: #f8f9ff;">
                                    <div class="table-responsive">
                                        <table class="table table-hover mb-0 table-sm" id="paymentDistributionTable">
                                            <thead class="bg-primary text-white">
                                                <tr>
                                                    <th class="border-0 small">Date</th>
                                                    <th class="border-0 small">Product</th>
                                                    <th class="border-0 small">Total Amount</th>
                                                    <th class="border-0 small">Paid Amount</th>
                                                    <th class="border-0 small">Remaining</th>
                                                    <th class="border-0 small">This Payment</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer border-top bg-light py-2">
                    <button type="button" class="btn btn-secondary btn-sm px-3" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Close
                    </button>
                    <button type="button" class="btn btn-primary btn-sm px-3" onclick="submitPayment()">
                        <i class="fas fa-save me-1"></i>Save Payment
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteConfirmModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Delete Supplier</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="deleteWarning" class="alert alert-warning" style="display: none;">
                        <i class="fas fa-exclamation-triangle"></i>
                        <strong>Warning:</strong> You can't delete supplier, You have Pending Payments To Supplier!!<br>
                        <span id="amountOwed" class="mt-2 d-block"></span>
                    </div>
                    <div id="deleteConfirmation">
                        <p>Are you sure you want to delete supplier: <strong id="deleteSupplierName"></strong>?</p>
                        <p class="text-danger">This action cannot be undone.</p>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-danger" id="deleteButton" onclick="deleteSupplier()">Delete</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Payment History Modal -->
    <div class="modal fade" id="paymentHistoryModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Payment History for <span id="phSupplierName"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form class="row g-3 mb-3" id="phFilterForm">
                        <div class="col-md-4">
                            <label class="form-label">Start Date</label>
                            <input type="date" name="start_date" class="form-control">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">End Date</label>
                            <input type="date" name="end_date" class="form-control">
                        </div>
                        <div class="col-md-4 d-flex align-items-end">
                            <button type="submit" class="btn btn-primary w-100">Search</button>
                        </div>
                    </form>
                    <div class="table-responsive">
                        <table class="table table-bordered" id="phTable">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Payment Amount</th>
                                    <th>Payment Method</th>
                                    <th>Reference</th>
                                    <th>Balance After</th>
                                </tr>
                            </thead>
                            <tbody id="phTableBody">
                                <!-- Payment history rows will be loaded here -->
                            </tbody>
                        </table>
                    </div>
                    <button class="btn btn-secondary mt-3" id="phPrintBtn"><i class="fas fa-print"></i> Print</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Print Options Modal -->
    <div class="modal fade" id="printOptionsModal" tabindex="-1" aria-labelledby="printOptionsModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="printOptionsModalLabel">Print Options for <span id="printSupplierName"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <button type="button" class="btn btn-primary w-100 mb-2" id="printReportBtn"><i class="fas fa-file-alt"></i> Print Supplier Report</button>
                    <button type="button" class="btn btn-info w-100" id="printPaymentHistoryBtn"><i class="fas fa-file-invoice-dollar"></i> Print Payment History</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Create New Receipt Modal -->
    <div class="modal fade" id="createReceiptModal" tabindex="-1" aria-labelledby="createReceiptModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="createReceiptModalLabel">Create New Receipt</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form method="post" class="needs-validation" novalidate id="createReceiptForm">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Supplier *</label>
                                <select class="form-select form-select-sm" name="supplier_id" required>
                                    <option value="">Select Supplier</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?php echo $supplier['id']; ?>"><?php echo htmlspecialchars($supplier['name']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Items *</label>
                                <table class="table table-bordered" id="itemsTable">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>Quantity</th>
                                            <th>Unit Price</th>
                                            <th>Batch Number</th>
                                            <th>Notes</th>
                                            <th>Serial Numbers</th>
                                            <th></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr>
                                            <td>
                                                <select class="form-select form-select-sm" name="product_id[]" required>
                                                    <option value="">Select Product</option>
                                                    <?php foreach ($products as $product): ?>
                                                        <option value="<?php echo $product['id']; ?>"><?php echo htmlspecialchars($product['name']); ?></option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </td>
                                            <td><input type="number" class="form-control form-control-sm" name="quantity[]" min="1" required></td>
                                            <td><input type="number" class="form-control form-control-sm" name="unit_price[]" step="0.01" min="0.01" required></td>
                                            <td><input type="text" class="form-control form-control-sm" name="batch_number[]" required></td>
                                            <td><input type="text" class="form-control form-control-sm" name="notes[]"></td>
                                            <td><input type="text" class="form-control form-control-sm" name="serial_numbers[]"></td>
                                            <td><button type="button" class="btn btn-danger btn-sm remove-item">&times;</button></td>
                                        </tr>
                                    </tbody>
                                </table>
                                <button type="button" class="btn btn-success btn-sm" id="addItemRow"><i class="fas fa-plus"></i> Add Another Item</button>
                            </div>
                        </div>
                        <div class="mt-4">
                            <button type="submit" name="receive_item" class="btn btn-primary">
                                <i class="fas fa-save"></i> Create Receipt
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Success Modal -->
    <div class="modal fade" id="successModal" tabindex="-1" aria-labelledby="successModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0 shadow-lg">
                <div class="modal-header bg-success text-white border-0">
                    <h5 class="modal-title d-flex align-items-center" id="successModalLabel">
                        <span class="me-2" style="font-size:2rem;">
                            <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round" class="feather feather-check-circle animate__animated animate__bounceIn"><circle cx="12" cy="12" r="10"/><path d="M9 12l2 2l4-4"/></svg>
                        </span>
                        Success
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body text-center py-4">
                    <p id="successModalMessage" class="mb-0 fs-5 text-success">Payment recorded successfully!</p>
                </div>
                <div class="modal-footer border-0 justify-content-center">
                    <button type="button" class="btn btn-success px-4" data-bs-dismiss="modal" onclick="window.location.reload();">OK</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Error Modal -->
    <div class="modal fade" id="errorModal" tabindex="-1" aria-labelledby="errorModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0 shadow-lg">
                <div class="modal-header bg-danger text-white border-0">
                    <h5 class="modal-title d-flex align-items-center" id="errorModalLabel">
                        <span class="me-2" style="font-size:2rem;">
                            <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round" class="feather feather-x-circle animate__animated animate__shakeX"><circle cx="12" cy="12" r="10"/><path d="M15 9l-6 6M9 9l6 6"/></svg>
                        </span>
                        Error
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body text-center py-4">
                    <p id="errorModalMessage" class="mb-0 fs-5 text-danger">An error occurred.</p>
                </div>
                <div class="modal-footer border-0 justify-content-center">
                    <button type="button" class="btn btn-danger px-4" data-bs-dismiss="modal">OK</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Suppliers data for JavaScript
        const suppliers = <?php echo json_encode($suppliers); ?>;
        
        // Form validation
        (function () {
            'use strict'
            var forms = document.querySelectorAll('.needs-validation')
            Array.prototype.slice.call(forms).forEach(function (form) {
                form.addEventListener('submit', function (event) {
                    if (!form.checkValidity()) {
                        event.preventDefault()
                        event.stopPropagation()
                    }
                    form.classList.add('was-validated')
                }, false)
            })
        })()

        // Search functionality
        document.getElementById('supplierSearch').addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('.supplier-header');
            
            rows.forEach(row => {
                const name = row.querySelector('td:first-child').textContent.toLowerCase();
                const contact = row.querySelector('td:nth-child(2)').textContent.toLowerCase();
                const phone = row.querySelector('td:nth-child(2) small').textContent.toLowerCase();
                
                if (name.includes(searchTerm) || contact.includes(searchTerm) || phone.includes(searchTerm)) {
                    row.style.display = '';
                    // Also show/hide the details row if it exists
                    const detailsRow = document.querySelector(`.supplier-details[data-supplier-id="${row.dataset.supplierId}"]`);
                    if (detailsRow) {
                        detailsRow.style.display = row.classList.contains('expanded') ? 'table-row' : 'none';
                    }
                } else {
                    row.style.display = 'none';
                    // Hide the details row if it exists
                    const detailsRow = document.querySelector(`.supplier-details[data-supplier-id="${row.dataset.supplierId}"]`);
                    if (detailsRow) {
                        detailsRow.style.display = 'none';
                    }
                }
            });
        });

        // Handle edit supplier
        document.querySelectorAll('.edit-supplier').forEach(button => {
            button.addEventListener('click', function() {
                const supplier = JSON.parse(this.dataset.supplier);
                document.getElementById('supplier_id').value = supplier.id;
                document.getElementById('name').value = supplier.name;
                document.getElementById('contact_person').value = supplier.contact_person;
                document.getElementById('phone').value = supplier.phone;
                document.getElementById('email').value = supplier.email;
                document.getElementById('address').value = supplier.address;
                document.getElementById('tax_number').value = supplier.tax_number;
                document.getElementById('supplierModalLabel').textContent = 'Edit Supplier';
            });
        });

        // Reset form when modal is closed
        document.getElementById('supplierModal').addEventListener('hidden.bs.modal', function() {
            const form = this.querySelector('form');
            form.reset();
            form.classList.remove('was-validated');
            document.getElementById('supplier_id').value = '';
            document.getElementById('supplierModalLabel').textContent = 'Add Supplier';
        });

        // Add this to your existing JavaScript
        document.addEventListener('DOMContentLoaded', function() {
            // Add animation to payment method cards
            const paymentMethod = document.getElementById('payment_method');
            if (paymentMethod) {
                paymentMethod.addEventListener('change', function() {
                    const mpesaSection = document.getElementById('mpesaSection');
                    const chequeSection = document.getElementById('chequeSection');
                    
                    // Reset sections
                    if (mpesaSection) mpesaSection.style.display = 'none';
                    if (chequeSection) chequeSection.style.display = 'none';
                    
                    // Show selected section with animation
                    if (this.value === 'mpesa' && mpesaSection) {
                        mpesaSection.style.display = 'block';
                        mpesaSection.style.opacity = '0';
                    setTimeout(() => {
                            mpesaSection.style.opacity = '1';
                        }, 50);
                    } else if (this.value === 'cheque' && chequeSection) {
                        chequeSection.style.display = 'block';
                        chequeSection.style.opacity = '0';
                        setTimeout(() => {
                            chequeSection.style.opacity = '1';
                        }, 50);
                    }
                });
            }
            
            // Allow unrestricted input for payment amount
            const paymentAmount = document.getElementById('payment_amount');
            if (paymentAmount) {
                paymentAmount.addEventListener('input', function() {
                    // Remove invalid state when user types
                    this.classList.remove('is-invalid');
                });
                
                // Use change event for formatting and distribution updates to avoid interfering with typing
                paymentAmount.addEventListener('change', function() {
                    // Format to 2 decimal places when user finishes editing
                    if (this.value && !isNaN(this.value)) {
                        const value = parseFloat(this.value);
                        if (value > 0) {
                            this.value = value.toFixed(2);
                        }
                    }
                    
                    // Update payment distribution when amount changes
                    updatePaymentDistribution();
                });
                
                // Also update on blur to catch tab navigation
                paymentAmount.addEventListener('blur', function() {
                    if (this.value && !isNaN(this.value)) {
                        const value = parseFloat(this.value);
                        if (value > 0) {
                            this.value = value.toFixed(2);
                        }
                    }
                });
            }

            // Attach event listeners to update the preview in real time
            const fields = ['payment_amount', 'payment_method', 'mpesa_code', 'cheque_number'];
            fields.forEach(id => {
                const el = document.getElementById(id);
                if (el) {
                    el.addEventListener('input', updateSendingMessagePreview);
                    el.addEventListener('change', updateSendingMessagePreview);
                }
            });
        });

        let unpaidItems = [];
        let paymentModal;

        document.addEventListener('DOMContentLoaded', function() {
            // Initialize the payment modal
            paymentModal = new bootstrap.Modal(document.getElementById('paymentModal'));

            // Add click handlers for supplier rows (expand/collapse)
            document.querySelectorAll('.supplier-header').forEach(row => {
                row.addEventListener('click', function(e) {
                    // Don't trigger if clicking on buttons in the actions column
                    if (e.target.closest('.btn-group') || e.target.closest('button')) {
                        return;
                    }
                    
                    const supplierId = this.dataset.supplierId;
                    const detailsRow = document.querySelector(`.supplier-details[data-supplier-id="${supplierId}"]`);
                    const chevron = this.querySelector('.supplier-toggle');
                    
                    if (detailsRow) {
                        if (detailsRow.style.display === 'none' || detailsRow.style.display === '') {
                            // Expand
                            detailsRow.style.display = 'table-row';
                            this.classList.add('expanded');
                            if (chevron) {
                                chevron.classList.remove('fa-chevron-right');
                                chevron.classList.add('fa-chevron-down');
                            }
                        } else {
                            // Collapse
                            detailsRow.style.display = 'none';
                            this.classList.remove('expanded');
                            if (chevron) {
                                chevron.classList.remove('fa-chevron-down');
                                chevron.classList.add('fa-chevron-right');
                            }
                        }
                    }
                });
            });

            // Add click handlers for pay buttons
            document.querySelectorAll('.pay-btn').forEach(button => {
                button.addEventListener('click', function(e) {
                    e.stopPropagation();
                    const supplierId = this.dataset.supplierId;
                    const supplierName = this.dataset.supplierName;
                    const totalAmount = parseFloat(this.dataset.totalAmount);
                    const paidAmount = parseFloat(this.dataset.paidAmount);
                    const items = JSON.parse(this.dataset.items);
                    
                    showPaymentModal(supplierId, supplierName, totalAmount, paidAmount, items);
                });
            });
        });

        function showPaymentModal(supplierId, supplierName, totalAmount, paidAmount, items) {
            try {
                // Ensure numbers
                totalAmount = parseFloat(totalAmount) || 0;
                paidAmount = parseFloat(paidAmount) || 0;
                const balance = totalAmount - paidAmount;
                
                // Debug console output
                console.log('=== TOTAL BALANCE DEBUG ===');
                console.log('Supplier ID:', supplierId);
                console.log('Supplier Name:', supplierName);
                console.log('Total Amount:', totalAmount);
                console.log('Paid Amount:', paidAmount);
                console.log('Calculated Balance:', balance);
                console.log('Items received:', items);
                
                // Check individual item calculations
                let calculatedTotal = 0;
                let calculatedPaid = 0;
                items.forEach((item, index) => {
                    const itemTotal = parseFloat(item.quantity) * parseFloat(item.unit_price);
                    calculatedTotal += itemTotal;
                    
                                            if (item.payment_status === 'paid') {
                        calculatedPaid += itemTotal;
                    }
                    
                    console.log(`Item ${index + 1}:`, {
                        product: item.product_name,
                        quantity: item.quantity,
                        unit_price: item.unit_price,
                        total: itemTotal,
                        payment_status: item.payment_status,
                        is_paid: (item.payment_status === 'paid')
                    });
                });
                
                console.log('Calculated Total from Items:', calculatedTotal);
                console.log('Calculated Paid from Items:', calculatedPaid);
                console.log('Calculated Balance from Items:', calculatedTotal - calculatedPaid);
                console.log('=== END DEBUG ===');

                // Store original balance for calculations
                window.originalBalance = balance;

                // Filter unpaid items
                unpaidItems = items.filter(item => {
                    const total = parseFloat(item.quantity) * parseFloat(item.unit_price);
                    const isPaid = item.payment_status === 'paid';
                    const paidAmount = isPaid ? total : 0;
                    item.total = total;
                    item.paid = paidAmount;
                    item.remaining = total - paidAmount;
                    
                    // Ensure the id property is set correctly
                    if (!item.id) {
                        console.error('Item missing id property:', item);
                    }
                    
                    return item.remaining > 0;
                });
                
                console.log('Filtered unpaid items:', unpaidItems);

                // Reset the form first
                const paymentForm = document.getElementById('paymentForm');
                if (paymentForm) paymentForm.reset();
                
                const mpesaSection = document.getElementById('mpesaSection');
                if (mpesaSection) mpesaSection.style.display = 'none';
                
                const chequeSection = document.getElementById('chequeSection');
                if (chequeSection) chequeSection.style.display = 'none';

                // Now set the values in the modal
                const supplierIdInput = document.getElementById('payment_supplier_id');
                if (supplierIdInput) supplierIdInput.value = supplierId;
                
                const totalBalanceElement = document.getElementById('total_balance');
                if (totalBalanceElement) totalBalanceElement.textContent = balance.toFixed(2);
                
                // Set initial remaining balance to same as total balance
                const remainingBalanceElement = document.getElementById('remaining_balance');
                if (remainingBalanceElement) remainingBalanceElement.textContent = balance.toFixed(2);
                
                // Update balance calculation display
                const balanceCalculationElement = document.getElementById('balance_calculation');
                if (balanceCalculationElement) {
                    balanceCalculationElement.innerHTML = `
                        Total: ${totalAmount.toFixed(2)} - Paid: ${paidAmount.toFixed(2)} = ${balance.toFixed(2)}
                    `;
                }
                
                const paymentAmountInput = document.getElementById('payment_amount');
                if (paymentAmountInput) {
                    paymentAmountInput.removeAttribute('max'); // Allow any amount
                    paymentAmountInput.removeAttribute('min'); // Remove minimum restriction
                    paymentAmountInput.value = balance.toFixed(2);
                }

                // Update payment distribution table
                updatePaymentDistributionTable();

                // Automatically distribute the payment amount across unpaid items
                updatePaymentDistribution();

                // Check if supplier has contact information for notifications
                const supplier = suppliers.find(s => s.id == supplierId);
                if (supplier) {
                    const hasPhone = supplier.phone && supplier.phone.trim() !== '';
                    const hasEmail = supplier.email && supplier.email.trim() !== '';
                    
                    // Get notification settings
                    const smsEnabled = window.notificationSettings ? window.notificationSettings.enableSMS === '1' : false;
                    const whatsappEnabled = window.notificationSettings ? window.notificationSettings.enableWhatsApp === '1' : false;
                    const emailEnabled = window.notificationSettings ? window.notificationSettings.enableEmail === '1' : false;
                    
                    // Show/hide notification options based on available contact info AND settings
                    const sendSmsCheckbox = document.getElementById('send_sms');
                    const sendWhatsappCheckbox = document.getElementById('send_whatsapp');
                    const sendEmailCheckbox = document.getElementById('send_email');
                    
                    // SMS: Check both settings and phone availability
                    if (sendSmsCheckbox) {
                        const shouldDisable = !hasPhone || !smsEnabled;
                        sendSmsCheckbox.disabled = shouldDisable;
                        sendSmsCheckbox.checked = false; // Always uncheck when disabled
                        
                        // Add visual styling
                        const smsCard = sendSmsCheckbox.closest('.card');
                        if (smsCard) {
                            if (shouldDisable) {
                                smsCard.classList.add('notification-card-disabled');
                            } else {
                                smsCard.classList.remove('notification-card-disabled');
                            }
                        }
                    }
                    
                    // WhatsApp: Check both settings and phone availability
                    if (sendWhatsappCheckbox) {
                        const shouldDisable = !hasPhone || !whatsappEnabled;
                        sendWhatsappCheckbox.disabled = shouldDisable;
                        sendWhatsappCheckbox.checked = false; // Always uncheck when disabled
                        
                        // Add visual styling for WhatsApp (it's in the same card as email)
                        const whatsappCard = sendWhatsappCheckbox.closest('.card');
                        if (whatsappCard) {
                            const emailShouldDisable = !hasEmail || !emailEnabled;
                            const anyEnabled = (!shouldDisable || !emailShouldDisable);
                            
                            if (!anyEnabled) {
                                whatsappCard.classList.add('notification-card-disabled');
                            } else {
                                whatsappCard.classList.remove('notification-card-disabled');
                            }
                        }
                    }
                    
                    // Email: Check both settings and email availability
                    if (sendEmailCheckbox) {
                        const shouldDisable = !hasEmail || !emailEnabled;
                        sendEmailCheckbox.disabled = shouldDisable;
                        sendEmailCheckbox.checked = false; // Always uncheck when disabled
                        
                        // Visual styling is handled in WhatsApp section since they're in the same card
                    }
                    
                    // Add visual indicators with more detailed messages
                    const smsLabel = document.querySelector('label[for="send_sms"]');
                    const whatsappLabel = document.querySelector('label[for="send_whatsapp"]');
                    const emailLabel = document.querySelector('label[for="send_email"]');
                    
                    if (smsLabel) {
                        if (!hasPhone) {
                            smsLabel.innerHTML = '<i class="fas fa-sms text-muted me-1"></i>SMS Notification <small class="text-muted">(No phone number)</small>';
                        } else if (!smsEnabled) {
                            smsLabel.innerHTML = '<i class="fas fa-sms text-muted me-1"></i>SMS Notification <small class="text-muted">(Disabled in settings)</small>';
                        } else {
                            smsLabel.innerHTML = '<i class="fas fa-sms text-success me-1"></i>SMS Notification';
                        }
                    }
                    
                    if (whatsappLabel) {
                        if (!hasPhone) {
                            whatsappLabel.innerHTML = '<i class="fab fa-whatsapp text-muted me-1"></i>WhatsApp Message <small class="text-muted">(No phone number)</small>';
                        } else if (!whatsappEnabled) {
                            whatsappLabel.innerHTML = '<i class="fab fa-whatsapp text-muted me-1"></i>WhatsApp Message <small class="text-muted">(Disabled in settings)</small>';
                        } else {
                            whatsappLabel.innerHTML = '<i class="fab fa-whatsapp text-success me-1"></i>WhatsApp Message';
                        }
                    }
                    
                    if (emailLabel) {
                        if (!hasEmail) {
                            emailLabel.innerHTML = '<i class="fas fa-envelope text-muted me-1"></i>Email Notification <small class="text-muted">(No email address)</small>';
                        } else if (!emailEnabled) {
                            emailLabel.innerHTML = '<i class="fas fa-envelope text-muted me-1"></i>Email Notification <small class="text-muted">(Disabled in settings)</small>';
                        } else {
                            emailLabel.innerHTML = '<i class="fas fa-envelope text-primary me-1"></i>Email Notification';
                        }
                    }
                }

                // Fetch previous messages for this supplier
                fetchPreviousMessages(supplierId);
                updateSendingMessagePreview();

                // Show the modal
                if (paymentModal) {
                    paymentModal.show();
                } else {
                    console.error('Payment modal not initialized');
                }
            } catch (error) {
                console.error('Error in showPaymentModal:', error);
                alert('Error loading payment details. Please try again.');
            }
        }

        function handlePaymentMethodChange() {
            const method = document.getElementById('payment_method').value;
            const mpesaSection = document.getElementById('mpesaSection');
            const chequeSection = document.getElementById('chequeSection');
            
            mpesaSection.style.display = method === 'mpesa' ? 'block' : 'none';
            chequeSection.style.display = method === 'cheque' ? 'block' : 'none';
            
            // Make fields required based on payment method
            document.getElementById('mpesa_code').required = method === 'mpesa';
            document.getElementById('cheque_number').required = method === 'cheque';
        }



        // Real-time payment preview function (fires on every keystroke)
        function updatePaymentPreview() {
            try {
                const paymentAmount = parseFloat(document.getElementById('payment_amount').value) || 0;
                const totalBalanceElement = document.getElementById('total_balance');
                
                if (!totalBalanceElement || !window.originalBalance) return;
                
                // Update Total Balance display to show remaining balance after payment
                const originalBalance = window.originalBalance;
                const remainingBalance = Math.max(0, originalBalance - paymentAmount);
                totalBalanceElement.textContent = remainingBalance.toFixed(2);
                
                // Update balance calculation details
                const balanceCalculation = document.getElementById('balance_calculation');
                if (balanceCalculation) {
                    if (paymentAmount > 0) {
                        balanceCalculation.textContent = `${originalBalance.toFixed(2)} - ${paymentAmount.toFixed(2)} = ${remainingBalance.toFixed(2)}`;
                    } else {
                        balanceCalculation.textContent = `Total amount owed: ${originalBalance.toFixed(2)}`;
                    }
                }
                
                // Update payment distribution in real-time
                updatePaymentDistributionPreview(paymentAmount);
                
                // Update message preview
                updateSendingMessagePreview();
                
            } catch (error) {
                console.error('Error in updatePaymentPreview:', error);
            }
        }

        // Update payment distribution preview (real-time)
        function updatePaymentDistributionPreview(paymentAmount) {
            if (!unpaidItems || unpaidItems.length === 0) return;
            
            let remainingPayment = paymentAmount;
            
            // Reset and distribute payment amounts
            const paymentCells = document.querySelectorAll('.payment-amount');
            unpaidItems.forEach((item, index) => {
                if (paymentCells[index]) {
                    if (remainingPayment <= 0) {
                        paymentCells[index].textContent = '0.00';
                    } else {
                        const itemPayment = Math.min(remainingPayment, item.remaining);
                        paymentCells[index].textContent = itemPayment.toFixed(2);
                        remainingPayment -= itemPayment;
                    }
                }
            });
        }

        function updatePaymentDistribution() {
            try {
                const paymentAmount = parseFloat(document.getElementById('payment_amount').value) || 0;
                const totalBalanceElement = document.getElementById('total_balance');
                const originalBalance = window.originalBalance || parseFloat(totalBalanceElement ? totalBalanceElement.textContent : '0') || 0;
                let remainingPayment = paymentAmount;
                
                // Reset all payment amounts
                document.querySelectorAll('.payment-amount').forEach(cell => {
                    cell.textContent = '0.00';
                });
                
                // Distribute payment across items
                unpaidItems.forEach((item, index) => {
                    if (remainingPayment <= 0) return;
                    
                    const paymentAmount = Math.min(remainingPayment, item.remaining);
                    remainingPayment -= paymentAmount;
                    
                    const cells = document.querySelectorAll('.payment-amount');
                    if (cells[index]) {
                        cells[index].textContent = paymentAmount.toFixed(2);
                    }
                });
                
                // Don't modify the payment amount input - let user control it
                // The distribution will be calculated based on what the user enters
                
                // Update the sending message preview
                updateSendingMessagePreview();
            } catch (error) {
                console.error('Error in updatePaymentDistribution:', error);
            }
        }

        function updatePaymentDistributionTable() {
            const tbody = document.querySelector('#paymentDistributionTable tbody');
            tbody.innerHTML = '';
            
            if (!unpaidItems || unpaidItems.length === 0) {
                const row = document.createElement('tr');
                row.innerHTML = '<td colspan="6" class="text-center">No unpaid items found</td>';
                tbody.appendChild(row);
                return;
            }
            
            unpaidItems.forEach(item => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${item.created_at}</td>
                    <td>${item.product_name}</td>
                    <td>${item.total.toFixed(2)}</td>
                    <td>${item.paid.toFixed(2)}</td>
                    <td>${item.remaining.toFixed(2)}</td>
                    <td class="payment-amount">0.00</td>
                `;
                tbody.appendChild(row);
            });
        }

        function submitPayment() {
            try {
                const form = document.getElementById('paymentForm');
                const formData = new FormData(form);
                
                // Validate payment amount
                const amount = parseFloat(formData.get('amount'));
                const totalBalanceElement = document.getElementById('total_balance');
                
                if (!totalBalanceElement) {
                    console.error('Total balance element not found');
                    const errorModalMessage = document.getElementById('errorModalMessage');
                    if (errorModalMessage) {
                        errorModalMessage.textContent = 'Error: Payment form not properly initialized';
                        var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                        errorModal.show();
                    } else {
                        alert('Error: Payment form not properly initialized');
                    }
                    return;
                }
                
                const totalBalance = parseFloat(totalBalanceElement.textContent);
                
                // Allow any amount to be entered (removed balance validation)
                // Only validate that amount is positive
                if (amount <= 0) {
                    const errorModalMessage = document.getElementById('errorModalMessage');
                    if (errorModalMessage) {
                        errorModalMessage.textContent = 'Please enter a valid payment amount greater than 0';
                        var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                        errorModal.show();
                    } else {
                        alert('Please enter a valid payment amount greater than 0');
                    }
                    document.getElementById('payment_amount').classList.add('is-invalid');
                    return;
                }
                
                // Validate required fields based on payment method
                const method = formData.get('payment_method');
                if (!method) {
                    const errorModalMessage = document.getElementById('errorModalMessage');
                    if (errorModalMessage) {
                        errorModalMessage.textContent = 'Please select a payment method';
                        var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                        errorModal.show();
                    } else {
                        alert('Please select a payment method');
                    }
                    return;
                }
                
                if (method === 'mpesa' && !formData.get('mpesa_code')) {
                    const errorModalMessage = document.getElementById('errorModalMessage');
                    if (errorModalMessage) {
                        errorModalMessage.textContent = 'Please enter M-PESA transaction code';
                        var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                        errorModal.show();
                    } else {
                        alert('Please enter M-PESA transaction code');
                    }
                    return;
                }
                if (method === 'cheque' && !formData.get('cheque_number')) {
                    const errorModalMessage = document.getElementById('errorModalMessage');
                    if (errorModalMessage) {
                        errorModalMessage.textContent = 'Please enter cheque number';
                        var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                        errorModal.show();
                    } else {
                        alert('Please enter cheque number');
                    }
                    return;
                }
                
                // Add payment distribution to form data
                const paymentDistribution = [];
                console.log('Building payment distribution...');
                console.log('Unpaid items:', unpaidItems);
                
                document.querySelectorAll('#paymentDistributionTable tbody tr').forEach((row, index) => {
                    const paymentCell = row.querySelector('.payment-amount');
                    const amount = parseFloat(paymentCell ? paymentCell.textContent : '0');
                    
                    console.log(`Row ${index}: amount=${amount}, unpaidItems[${index}]=`, unpaidItems[index]);
                    
                    if (amount > 0 && unpaidItems[index] && unpaidItems[index].id) {
                        paymentDistribution.push({
                            item_id: unpaidItems[index].id,
                            amount: amount
                        });
                        console.log(`Added to distribution: item_id=${unpaidItems[index].id}, amount=${amount}`);
                    }
                });
                
                console.log('Final payment distribution:', paymentDistribution);
                
                if (paymentDistribution.length === 0) {
                    console.error('Payment distribution is empty!');
                    console.log('Payment amount input value:', document.getElementById('payment_amount').value);
                    console.log('Payment distribution table rows:', document.querySelectorAll('#paymentDistributionTable tbody tr').length);
                    
                    const errorModalMessage = document.getElementById('errorModalMessage');
                    if (errorModalMessage) {
                        errorModalMessage.textContent = 'Payment distribution failed. Please check the payment amount and try again.';
                        var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                        errorModal.show();
                    } else {
                        alert('Payment distribution failed. Please check the payment amount and try again.');
                    }
                    return;
                }
                
                // Always send payment_status
                formData.append('payment_status', 'paid');
                try {
                formData.append('payment_distribution', JSON.stringify(paymentDistribution));
                } catch (error) {
                    console.error('Error stringifying payment distribution:', error);
                    alert('Error processing payment data. Please try again.');
                    return;
                }
                
                // Show loading state
                const submitButton = document.querySelector('#paymentModal .modal-footer .btn-primary');
                const originalText = submitButton.innerHTML;
                submitButton.disabled = true;
                submitButton.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Processing...';
                
                // Submit payment
                fetch('process_payment.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        // Show Bootstrap success modal
                        const successModalMessage = document.getElementById('successModalMessage');
                        if (successModalMessage) {
                            successModalMessage.textContent = 'Payment recorded successfully!';
                            var successModal = new bootstrap.Modal(document.getElementById('successModal'));
                            successModal.show();
                            
                            // Reload the page after 2 seconds to show updated totals
                            setTimeout(() => {
                                window.location.reload();
                            }, 2000);
                        } else {
                            alert('Payment recorded successfully!');
                            // Reload the page to show updated totals
                            window.location.reload();
                        }
                    } else {
                        const errorModalMessage = document.getElementById('errorModalMessage');
                        if (errorModalMessage) {
                            errorModalMessage.textContent = data.message || 'An error occurred while processing the payment.';
                            var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                            errorModal.show();
                        } else {
                            alert(data.message || 'An error occurred while processing the payment.');
                        }
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    const errorModalMessage = document.getElementById('errorModalMessage');
                    if (errorModalMessage) {
                        errorModalMessage.textContent = 'Error: ' + error.message;
                        var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                        errorModal.show();
                    } else {
                        alert('Error: ' + error.message);
                    }
                })
                .finally(() => {
                    // Reset button state
                    submitButton.disabled = false;
                    submitButton.innerHTML = originalText;
                });
            } catch (error) {
                console.error('Error in submitPayment:', error);
                alert('An error occurred while processing the payment. Please try again.');
            }
        }

        function editSupplier(supplier) {
            document.getElementById('supplier_id').value = supplier.id;
            document.getElementById('name').value = supplier.name;
            document.getElementById('contact_person').value = supplier.contact_person;
            document.getElementById('phone').value = supplier.phone;
            document.getElementById('email').value = supplier.email;
            document.getElementById('address').value = supplier.address;
            document.getElementById('tax_number').value = supplier.tax_number;
            document.getElementById('supplierModalLabel').textContent = 'Edit Supplier';
            
            // Open the modal
            const modal = new bootstrap.Modal(document.getElementById('supplierModal'));
            modal.show();
        }

        let supplierToDelete = null;

        function confirmDeleteSupplier(supplierId, supplierName, totalAmount, paidAmount) {
            supplierToDelete = supplierId;
            
            // Get all required elements with null checks
            const deleteSupplierNameElement = document.getElementById('deleteSupplierName');
            const deleteConfirmModalElement = document.getElementById('deleteConfirmModal');
            const warningDiv = document.getElementById('deleteWarning');
            const confirmationDiv = document.getElementById('deleteConfirmation');
            const deleteButton = document.getElementById('deleteButton');
            const amountOwedElement = document.getElementById('amountOwed');
            
            // Check if all required elements exist
            if (!deleteSupplierNameElement || !deleteConfirmModalElement || !warningDiv || !confirmationDiv || !deleteButton) {
                console.error('Required delete modal elements not found');
                alert('Error: Delete modal not properly initialized');
                return;
            }
            
            deleteSupplierNameElement.textContent = supplierName;
            
            const modal = new bootstrap.Modal(deleteConfirmModalElement);
            
            // Check if both items and amounts are fully paid
            const isFullyPaid = paidAmount >= totalAmount;
            
            if (!isFullyPaid) {
                // Calculate amount owed
                const amountOwed = totalAmount - paidAmount;
                
                // Update warning message with amount owed (only if element exists)
                if (amountOwedElement) {
                    amountOwedElement.innerHTML = 
                        `<strong>Amount Owed:</strong> ${amountOwed.toFixed(2)}`;
                }
                
                warningDiv.style.display = 'block';
                confirmationDiv.style.display = 'none';
                deleteButton.style.display = 'none';
            } else {
                warningDiv.style.display = 'none';
                confirmationDiv.style.display = 'block';
                deleteButton.style.display = 'block';
            }
            
            modal.show();
        }

        function deleteSupplier() {
            if (!supplierToDelete) return;

            const formData = new FormData();
            formData.append('supplier_id', supplierToDelete);
            formData.append('delete_supplier', '1');

            fetch('suppliers.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(() => {
                window.location.reload();
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while deleting the supplier.');
            });
        }

        let currentPrintSupplierId = null;
        let currentPrintSupplierName = '';
        function openPrintModal(supplierId, supplierName) {
            currentPrintSupplierId = supplierId;
            currentPrintSupplierName = supplierName;
            document.getElementById('printSupplierName').textContent = supplierName;
            new bootstrap.Modal(document.getElementById('printOptionsModal')).show();
        }
        document.getElementById('printReportBtn').onclick = function() {
            if (currentPrintSupplierId) {
                window.open('print_supplier_report.php?supplier_id=' + currentPrintSupplierId, '_blank');
            }
        };
        document.getElementById('printPaymentHistoryBtn').onclick = function() {
            if (currentPrintSupplierId) {
                window.open('supplier_payment_history.php?supplier_id=' + currentPrintSupplierId + '&print=1', '_blank');
            }
        };

        document.getElementById('addItemRow').addEventListener('click', function() {
            const table = document.getElementById('itemsTable').querySelector('tbody');
            const newRow = table.rows[0].cloneNode(true);
            newRow.querySelectorAll('input, select').forEach(input => input.value = '');
            table.appendChild(newRow);
        });
        document.getElementById('itemsTable').addEventListener('click', function(e) {
            if (e.target.classList.contains('remove-item')) {
                const row = e.target.closest('tr');
                if (row.parentNode.rows.length > 1) row.remove();
            }
        });

        // Function to update the sending message preview
        function updateSendingMessagePreview() {
            const supplierIdInput = document.getElementById('payment_supplier_id');
            const paymentAmountInput = document.getElementById('payment_amount');
            const paymentMethodSelect = document.getElementById('payment_method');
            const mpesaCodeInput = document.getElementById('mpesa_code');
            const chequeNumberInput = document.getElementById('cheque_number');
            const totalBalanceElement = document.getElementById('total_balance');
            const sendingMessagePreviewElement = document.getElementById('sendingMessagePreview');
            
            if (!supplierIdInput || !paymentAmountInput || !sendingMessagePreviewElement) {
                return; // Required elements don't exist
            }
            
            const supplierId = supplierIdInput.value;
            const amount = parseFloat(paymentAmountInput.value) || 0;
            const paymentMethod = paymentMethodSelect ? paymentMethodSelect.value : '';
            const mpesaCode = mpesaCodeInput ? mpesaCodeInput.value.trim() : '';
            const chequeNumber = chequeNumberInput ? chequeNumberInput.value.trim() : '';
            const supplier = suppliers.find(s => s.id == supplierId);
            const contactPerson = supplier && supplier.contact_person ? supplier.contact_person : (supplier ? supplier.name : 'Supplier');
            const storeName = window.storeName || 'Our Store';
            const storePhone = window.storePhone || '';
            const storeEmail = window.storeEmail || '';
            const currency = window.storeCurrency || 'Ksh';
            const balance = totalBalanceElement ? totalBalanceElement.textContent : '';

            let msg = '';
            if (!amount || amount <= 0) {
                sendingMessagePreviewElement.innerHTML = '<i class="fas fa-info-circle me-1"></i>Fill in payment details to preview the message.';
                return;
            }
            msg += `Dear ${contactPerson},\n\n`;
            msg += `A payment of ${currency} ${amount.toLocaleString(undefined, {minimumFractionDigits:2, maximumFractionDigits:2})} has been made to your account via `;
            if (paymentMethod === 'mpesa') {
                msg += `M-PESA${mpesaCode ? ` (Ref: ${mpesaCode})` : ''}. `;
            } else if (paymentMethod === 'cheque') {
                msg += `Cheque${chequeNumber ? ` (No: ${chequeNumber})` : ''}. `;
            } else if (paymentMethod) {
                msg += paymentMethod.charAt(0).toUpperCase() + paymentMethod.slice(1) + '. ';
            } else {
                msg += '. ';
            }
            if (balance && !isNaN(balance)) {
                // Use original balance for calculation instead of modified balance
                const originalBalance = window.originalBalance || parseFloat(balance);
                const remainingBalance = Math.max(0, originalBalance - amount);
                
                if (remainingBalance <= 0.01) {
                    msg += 'Your account is now fully paid. ';
                } else {
                    msg += `Remaining balance: ${currency} ${remainingBalance.toLocaleString(undefined, {minimumFractionDigits:2, maximumFractionDigits:2})}. `;
                }
            }
            msg += `\n\nThank you for your business!\n${storeName}`;
            if (storePhone) msg += `\nPhone: ${storePhone}`;
            if (storeEmail) msg += `\nEmail: ${storeEmail}`;

            sendingMessagePreviewElement.textContent = msg;
        }

        // Function to fetch previous messages for a supplier
        function fetchPreviousMessages(supplierId) {
            console.log('fetchPreviousMessages called with supplierId:', supplierId, 'type:', typeof supplierId);
            
            // Validate supplierId
            if (!supplierId || supplierId === 'undefined' || supplierId === 'null') {
                console.error('Invalid supplierId:', supplierId);
                const previousMessageContent = document.getElementById('previousMessageContent');
                if (previousMessageContent) {
                    previousMessageContent.innerHTML = `
                        <div class="text-muted text-center py-3">
                            <i class="fas fa-info-circle me-1"></i>
                            No previous messages found for this supplier.
                        </div>
                    `;
                }
                return;
            }
            
            fetch('../api/get_supplier_messages.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    supplier_id: supplierId
                })
            })
            .then(response => response.json())
            .then(data => {
                console.log('API response:', data);
                const previousMessageContent = document.getElementById('previousMessageContent');
                if (!previousMessageContent) return;
                
                if (data.success && data.messages && data.messages.length > 0) {
                    const latestMessage = data.messages[0];
                    const messageDate = new Date(latestMessage.created_at).toLocaleDateString();
                    const messageTime = new Date(latestMessage.created_at).toLocaleTimeString();
                    
                    previousMessageContent.innerHTML = `
                        <div class="border rounded p-3 bg-light">
                            <div class="d-flex justify-content-between align-items-start mb-2">
                                <small class="text-muted">
                                    <i class="fas fa-calendar me-1"></i>${messageDate} at ${messageTime}
                                </small>
                                <span class="badge bg-${getNotificationTypeColor(latestMessage.type)}">
                                    ${latestMessage.type.toUpperCase()}
                                </span>
                            </div>
                            <div class="message-content">
                                <strong>Amount:</strong> KES ${parseFloat(latestMessage.amount).toLocaleString()}<br>
                                <strong>Message:</strong> ${latestMessage.message}
                            </div>
                        </div>
                    `;
                } else {
                    previousMessageContent.innerHTML = `
                        <div class="text-muted text-center py-3">
                            <i class="fas fa-info-circle me-1"></i>
                            No previous messages found for this supplier.
                        </div>
                    `;
                }
            })
            .catch(error => {
                console.error('Error fetching previous messages:', error);
                const previousMessageContent = document.getElementById('previousMessageContent');
                if (previousMessageContent) {
                    previousMessageContent.innerHTML = `
                        <div class="text-danger text-center py-3">
                            <i class="fas fa-exclamation-triangle me-1"></i>
                            Error loading previous messages.
                        </div>
                    `;
                }
            });
        }

        // Helper function to get notification type color
        function getNotificationTypeColor(type) {
            switch(type) {
                case 'sms': return 'primary';
                case 'whatsapp': return 'success';
                case 'email': return 'info';
                default: return 'secondary';
            }
        }

        // Pass notification settings to JavaScript
        window.notificationSettings = {
            enableSMS: '<?php echo $notificationSettings['enable_sms'] ?? '0'; ?>',
            enableEmail: '<?php echo $notificationSettings['enable_email'] ?? '0'; ?>',
            enableWhatsApp: '<?php echo $notificationSettings['enable_whatsapp'] ?? '0'; ?>'
        };
    </script>
</body>
</html> 